<?php
require_once '../private/config.php';

// Oturum Kontrolü
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header("Location: login.php");
    exit;
}

$message = "";
$messageType = "";

// --- İŞLEM: SİLME ---
if (isset($_GET['delete'])) {
    $id = intval($_GET['delete']);
    $db->prepare("DELETE FROM indirimler WHERE id = ?")->execute([$id]);
    header("Location: urun-kategori-kampanya-ayar.php?msg=deleted");
    exit;
}

// --- İŞLEM: KAYDETME / GÜNCELLEME ---
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
    $kampanya_adi = trim($_POST['kampanya_adi']);
    $indirim_turu = $_POST['indirim_turu'];
    $indirim_degeri = floatval($_POST['indirim_degeri']);
    $hedef_tipi = $_POST['hedef_tipi'];
    
    // Çoklu seçim desteği (Gelen diziyi virgülle birleştirerek string yapıyoruz)
    $hedef_id_array = isset($_POST['hedef_id']) ? $_POST['hedef_id'] : [];
    $hedef_id = ($hedef_tipi == 'genel') ? null : (is_array($hedef_id_array) ? implode(',', $hedef_id_array) : $hedef_id_array);
    
    $baslangic = !empty($_POST['baslangic']) ? $_POST['baslangic'] : null;
    $bitis = !empty($_POST['bitis']) ? $_POST['bitis'] : null;
    $durum = intval($_POST['durum']);

    try {
        if ($id > 0) {
            $sql = "UPDATE indirimler SET kampanya_adi=?, indirim_turu=?, indirim_degeri=?, hedef_tipi=?, hedef_id=?, baslangic_tarihi=?, bitis_tarihi=?, durum=? WHERE id=?";
            $stmt = $db->prepare($sql);
            $stmt->execute([$kampanya_adi, $indirim_turu, $indirim_degeri, $hedef_tipi, $hedef_id, $baslangic, $bitis, $durum, $id]);
            $message = "Kampanya başarıyla güncellendi.";
        } else {
            $sql = "INSERT INTO indirimler (kampanya_adi, indirim_turu, indirim_degeri, hedef_tipi, hedef_id, baslangic_tarihi, bitis_tarihi, durum) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = $db->prepare($sql);
            $stmt->execute([$kampanya_adi, $indirim_turu, $indirim_degeri, $hedef_tipi, $hedef_id, $baslangic, $bitis, $durum]);
            $message = "Yeni kampanya başarıyla oluşturuldu.";
        }
        $messageType = "success";
    } catch (PDOException $e) {
        $message = "Hata: " . $e->getMessage();
        $messageType = "error";
    }
}

// --- VERİLERİ ÇEK ---
$campaigns = $db->query("SELECT * FROM indirimler ORDER BY id DESC")->fetchAll(PDO::FETCH_ASSOC);
$allCategories = $db->query("SELECT id, kategori_adi, ust_id FROM urunler_kategoriler ORDER BY sira ASC")->fetchAll(PDO::FETCH_ASSOC);
$allProducts = $db->query("SELECT id, urun_adi, urun_kodu FROM urunler ORDER BY urun_adi ASC")->fetchAll(PDO::FETCH_ASSOC);

// Lookup Map'leri oluştur (Liste ekranında isimleri göstermek için)
$catMap = []; foreach($allCategories as $c) $catMap[$c['id']] = $c['kategori_adi'];
$prodMap = []; foreach($allProducts as $p) $prodMap[$p['id']] = $p['urun_adi'];

// Kategori Hiyerarşisi Fonksiyonu
function getCategoriesHierarchy($categories, $parent = 0) {
    $result = [];
    foreach($categories as $cat) {
        if ($cat['ust_id'] == $parent) {
            $result[] = $cat;
        }
    }
    return $result;
}

// Düzenleme Verisi
$editData = null;
$selectedTargetIds = [];
if (isset($_GET['edit'])) {
    $id = intval($_GET['edit']);
    $stmt = $db->prepare("SELECT * FROM indirimler WHERE id = ?");
    $stmt->execute([$id]);
    $editData = $stmt->fetch(PDO::FETCH_ASSOC);
    if($editData && $editData['hedef_id']) {
        // Veritabanındaki "1,2,3" stringini diziye çeviriyoruz
        $selectedTargetIds = explode(',', $editData['hedef_id']);
    }
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kampanya Yönetimi | Panel</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="../assets/sidebar.css">
    <link rel="stylesheet" href="../assets/panel.css">
    <link rel="stylesheet" href="../assets/urun-kategori-kampanya-ayar.css">
</head>
<body>

<div class="admin-layout">
    <div class="overlay" id="sidebarOverlay"></div>
    <?php include 'sidebar.php'; ?>

    <main class="main-content">
        <header class="top-bar">
            <div style="display:flex; align-items:center;">
                <button id="toggleSidebar" class="hamburger"><i class="fa-solid fa-bars"></i></button>
                <h2>Kampanya Yönetimi</h2>
            </div>
            <div class="user-info">
                <span>Hoş geldin, <strong><?php echo $_SESSION['admin_username'] ?? 'Yönetici'; ?></strong></span>
            </div>
        </header>
        
        <div class="content-body">
            <?php if($message): ?>
                <div class="alert-box <?php echo $messageType; ?>">
                    <i class="fa-solid <?php echo $messageType == 'success' ? 'fa-check-circle' : 'fa-circle-exclamation'; ?>"></i>
                    <?php echo $message; ?>
                </div>
            <?php endif; ?>

            <div class="page-grid">
                
                <!-- KAMPANYA FORMU -->
                <div class="form-section">
                    <div class="panel-card">
                        <div class="card-header">
                            <h3><i class="fa-solid fa-bullhorn"></i> <?php echo $editData ? 'Düzenle' : 'Yeni Kampanya'; ?></h3>
                            <?php if($editData): ?><a href="urun-kategori-kampanya-ayar.php" class="btn-xs btn-outline">İptal</a><?php endif; ?>
                        </div>
                        <div class="card-body">
                            <form method="POST" action="" id="campaignForm">
                                <?php if($editData): ?><input type="hidden" name="id" value="<?php echo $editData['id']; ?>"><?php endif; ?>
                                
                                <div class="form-group">
                                    <label>Kampanya Adı</label>
                                    <input type="text" name="kampanya_adi" required value="<?php echo $editData['kampanya_adi'] ?? ''; ?>" placeholder="Örn: %20 Kış Fırsatı">
                                </div>

                                <div class="form-row">
                                    <div class="form-group col">
                                        <label>İndirim Türü</label>
                                        <select name="indirim_turu" class="form-select">
                                            <option value="oran" <?php echo (isset($editData) && $editData['indirim_turu'] == 'oran') ? 'selected' : ''; ?>>Yüzde (%)</option>
                                            <option value="tutar" <?php echo (isset($editData) && $editData['indirim_turu'] == 'tutar') ? 'selected' : ''; ?>>Sabit (₺)</option>
                                        </select>
                                    </div>
                                    <div class="form-group col">
                                        <label>Değer</label>
                                        <input type="number" step="0.01" name="indirim_degeri" required value="<?php echo $editData['indirim_degeri'] ?? ''; ?>">
                                    </div>
                                </div>

                                <div class="form-group">
                                    <label>Uygulama Hedefi</label>
                                    <select name="hedef_tipi" id="hedef_tipi" class="form-select" onchange="toggleTargetList(true)">
                                        <option value="genel" <?php echo (isset($editData) && $editData['hedef_tipi'] == 'genel') ? 'selected' : ''; ?>>Tüm Site Geneli</option>
                                        <option value="kategori" <?php echo (isset($editData) && $editData['hedef_tipi'] == 'kategori') ? 'selected' : ''; ?>>Seçili Kategoriler</option>
                                        <option value="urun" <?php echo (isset($editData) && $editData['hedef_tipi'] == 'urun') ? 'selected' : ''; ?>>Seçili Ürünler</option>
                                    </select>
                                </div>

                                <!-- KATEGORİ SEÇİMİ -->
                                <div id="category_select_area" class="target-select-box" style="display:none;">
                                    <div class="search-wrapper">
                                        <input type="text" onkeyup="filterList(this, 'cat_list')" placeholder="Kategori Ara...">
                                        <i class="fa-solid fa-magnifying-glass"></i>
                                    </div>
                                    <div class="checkbox-scroll" id="cat_list">
                                        <?php
                                        function buildCategoryTreeForm($categories, $parent=0, $indent='', $selected=[]) {
                                            $cats = getCategoriesHierarchy($categories, $parent);
                                            foreach($cats as $cat) {
                                                $check = in_array($cat['id'], $selected) ? 'checked' : '';
                                                echo '<label class="list-item" data-text="'.strtolower($cat['kategori_adi']).'">
                                                        <input type="checkbox" name="hedef_id[]" value="'.$cat['id'].'" '.$check.' class="cat-checkbox target-input" data-id="'.$cat['id'].'" data-parent="'.$cat['ust_id'].'">
                                                        <span>'.$indent.' '.$cat['kategori_adi'].'</span>
                                                      </label>';
                                                buildCategoryTreeForm($categories, $cat['id'], $indent . '--', $selected);
                                            }
                                        }
                                        buildCategoryTreeForm($allCategories, 0, '', $selectedTargetIds);
                                        ?>
                                    </div>
                                    <small class="helper-text">* Üst kategori seçilirse alt kırılımlar da kapsanır.</small>
                                </div>

                                <!-- ÜRÜN SEÇİMİ -->
                                <div id="product_select_area" class="target-select-box" style="display:none;">
                                    <div class="search-wrapper">
                                        <input type="text" onkeyup="filterList(this, 'prod_list')" placeholder="Ürün Ara (Ad veya Kod)...">
                                        <i class="fa-solid fa-magnifying-glass"></i>
                                    </div>
                                    <div class="checkbox-scroll" id="prod_list">
                                        <?php foreach($allProducts as $prod): ?>
                                            <?php $check = in_array($prod['id'], $selectedTargetIds) ? 'checked' : ''; ?>
                                            <label class="list-item" data-text="<?php echo strtolower($prod['urun_adi'].' '.$prod['urun_kodu']); ?>">
                                                <input type="checkbox" name="hedef_id[]" value="<?php echo $prod['id']; ?>" <?php echo $check; ?> class="target-input">
                                                <span><?php echo $prod['urun_adi']; ?> <small>(<?php echo $prod['urun_kodu']; ?>)</small></span>
                                            </label>
                                        <?php endforeach; ?>
                                    </div>
                                </div>

                                <div class="form-row">
                                    <div class="form-group col">
                                        <label>Başlangıç</label>
                                        <input type="datetime-local" name="baslangic" value="<?php echo isset($editData['baslangic_tarihi']) ? date('Y-m-d\TH:i', strtotime($editData['baslangic_tarihi'])) : ''; ?>">
                                    </div>
                                    <div class="form-group col">
                                        <label>Bitiş</label>
                                        <input type="datetime-local" name="bitis" value="<?php echo isset($editData['bitis_tarihi']) ? date('Y-m-d\TH:i', strtotime($editData['bitis_tarihi'])) : ''; ?>">
                                    </div>
                                </div>

                                <div class="form-group">
                                    <label>Kampanya Durumu</label>
                                    <select name="durum" class="form-select">
                                        <option value="1" <?php echo (!isset($editData) || $editData['durum'] == 1) ? 'selected' : ''; ?>>Aktif</option>
                                        <option value="0" <?php echo (isset($editData) && $editData['durum'] == 0) ? 'selected' : ''; ?>>Pasif</option>
                                    </select>
                                </div>

                                <button type="submit" class="save-btn btn-full">
                                    <i class="fa-solid fa-floppy-disk"></i> Ayarları Kaydet
                                </button>
                            </form>
                        </div>
                    </div>
                </div>

                <!-- LİSTE SÜTUNU -->
                <div class="list-section">
                    <div class="panel-card">
                        <div class="card-header">
                            <h3><i class="fa-solid fa-list-check"></i> Kampanya Listesi</h3>
                        </div>
                        <div class="card-body p-0">
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th>Kampanya</th>
                                        <th>Hedef & Detay</th>
                                        <th>İndirim</th>
                                        <th>Durum</th>
                                        <th width="80">İşlem</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach($campaigns as $camp): ?>
                                    <tr>
                                        <td><strong><?php echo $camp['kampanya_adi']; ?></strong></td>
                                        <td>
                                            <div style="display:flex; flex-direction:column; gap:5px;">
                                                <span class="target-badge <?php echo $camp['hedef_tipi']; ?>"><?php echo strtoupper($camp['hedef_tipi']); ?></span>
                                                <div class="target-labels">
                                                    <?php 
                                                    if($camp['hedef_tipi'] == 'genel') {
                                                        echo '<small class="text-muted">Tüm Site</small>';
                                                    } else {
                                                        $ids = array_filter(explode(',', $camp['hedef_id']));
                                                        $labels = [];
                                                        $map = ($camp['hedef_tipi'] == 'kategori') ? $catMap : $prodMap;
                                                        
                                                        $i = 0;
                                                        foreach($ids as $tid) {
                                                            if(isset($map[$tid])) {
                                                                $labels[] = $map[$tid];
                                                                $i++;
                                                            }
                                                            if($i >= 3) break;
                                                        }
                                                        
                                                        echo '<div class="scope-list">';
                                                        foreach($labels as $label) echo '<span class="scope-item">'.$label.'</span>';
                                                        if(count($ids) > 3) echo '<span class="scope-item more">+ '.(count($ids)-3).' daha</span>';
                                                        if(empty($labels)) echo '<small class="text-danger">Seçim bulunamadı</small>';
                                                        echo '</div>';
                                                    }
                                                    ?>
                                                </div>
                                            </div>
                                        </td>
                                        <td><?php echo $camp['indirim_turu'] == 'oran' ? '<span class="text-orange" style="font-weight:700;">%'.$camp['indirim_degeri'].'</span>' : '<span class="text-blue" style="font-weight:700;">'.number_format($camp['indirim_degeri'], 2).' ₺</span>'; ?></td>
                                        <td>
                                            <?php if($camp['durum'] == 1): ?>
                                                <span class="status-pill active">Yayında</span>
                                            <?php else: ?>
                                                <span class="status-pill passive">Pasif</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div class="action-buttons">
                                                <a href="?edit=<?php echo $camp['id']; ?>" class="btn-icon btn-edit"><i class="fa-solid fa-pen"></i></a>
                                                <a href="?delete=<?php echo $camp['id']; ?>" class="btn-icon btn-delete" onclick="return confirm('Silmek istediğine emin misin?')"><i class="fa-solid fa-trash"></i></a>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

            </div>
        </div>
    </main>
</div>

<script>
    /**
     * Hedef Tipi listelerini açar/kapatır
     * @param {boolean} clearOld Eğer tip elle değiştirildiyse diğer seçimleri temizler
     */
    function toggleTargetList(clearOld = false) {
        const type = document.getElementById('hedef_tipi').value;
        const catArea = document.getElementById('category_select_area');
        const prodArea = document.getElementById('product_select_area');

        // Önce hepsini gizle ve inputlarını disable et (Disable olan inputlar form ile gönderilmez)
        catArea.style.display = 'none';
        prodArea.style.display = 'none';
        catArea.querySelectorAll('input').forEach(i => i.disabled = true);
        prodArea.querySelectorAll('input').forEach(i => i.disabled = true);

        if(type === 'kategori') {
            catArea.style.display = 'block';
            catArea.querySelectorAll('input').forEach(i => i.disabled = false);
            if(clearOld) clearCheckboxes(prodArea); 
        } else if(type === 'urun') {
            prodArea.style.display = 'block';
            prodArea.querySelectorAll('input').forEach(i => i.disabled = false);
            if(clearOld) clearCheckboxes(catArea); 
        } else if(type === 'genel') {
            if(clearOld) {
                clearCheckboxes(catArea);
                clearCheckboxes(prodArea);
            }
        }
    }

    // Belirli bir alandaki tüm checkboxları temizle
    function clearCheckboxes(container) {
        container.querySelectorAll('input[type="checkbox"]').forEach(cb => {
            cb.checked = false;
        });
    }

    function filterList(input, listId) {
        const filter = input.value.toLowerCase();
        const items = document.querySelectorAll('#' + listId + ' .list-item');
        items.forEach(item => {
            const text = item.getAttribute('data-text');
            item.style.display = text.includes(filter) ? 'flex' : 'none';
        });
    }

    // Hiyerarşik Kategori Checkbox Mantığı
    document.querySelectorAll('.cat-checkbox').forEach(cb => {
        cb.addEventListener('change', function() {
            const id = this.getAttribute('data-id');
            const isChecked = this.checked;
            
            // Tüm alt çocukları bul ve işaretle
            document.querySelectorAll(`.cat-checkbox[data-parent="${id}"]`).forEach(child => {
                child.checked = isChecked;
                // Tetikleme yap ki torunlar da etkilensin
                child.dispatchEvent(new Event('change'));
            });
        });
    });

    document.addEventListener("DOMContentLoaded", () => {
        toggleTargetList(false); // Sayfa yüklendiğinde mevcut durumu aç (temizleme yapma)
    });
</script>

<style>
/* Kampanya Listesi Ekstra Tasarım */
.scope-list { display: flex; flex-wrap: wrap; gap: 4px; margin-top: 2px; }
.scope-item { 
    font-size: 0.75rem; 
    background: #f1f5f9; 
    color: #475569; 
    padding: 2px 6px; 
    border-radius: 4px; 
    border: 1px solid #e2e8f0;
    max-width: 150px;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}
.scope-item.more { background: #eff6ff; color: #2563eb; border-color: #dbeafe; font-weight: 600; }
.text-orange { color: #f97316; }
.text-blue { color: #2563eb; }
</style>

</body>
</html>